import { useRef, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { useClickOutside } from '@/hooks/useClickOutside';
import type { MegaMenuData } from './menuData';

interface MegaMenuProps {
  data: MegaMenuData;
  isOpen: boolean;
  onClose: () => void;
  triggerRef: React.RefObject<HTMLDivElement>;
  isHeaderSolid?: boolean;
}

const MegaMenu = ({ data, isOpen, onClose, triggerRef, isHeaderSolid = false }: MegaMenuProps) => {
  const menuRef = useRef<HTMLDivElement>(null);

  // Fechar ao clicar fora
  useClickOutside(menuRef, () => {
    if (isOpen) {
      onClose();
    }
  }, triggerRef);

  // Fechar ao pressionar ESC
  useEffect(() => {
    const handleEscape = (e: KeyboardEvent) => {
      if (e.key === 'Escape' && isOpen) {
        onClose();
      }
    };

    document.addEventListener('keydown', handleEscape);
    return () => document.removeEventListener('keydown', handleEscape);
  }, [isOpen, onClose]);


  if (!isOpen) return null;

  return (
    <div
      ref={menuRef}
      className="fixed left-1/2 -translate-x-1/2 top-[72px] w-[90vw] max-w-[1200px] z-50 pointer-events-auto"
      role="menu"
      aria-hidden={!isOpen}
    >
      <div
        className={`border border-white/10 rounded-2xl shadow-2xl p-8 transition-all duration-300 ${
          isHeaderSolid
            ? "bg-[#0E1B2E]/95 backdrop-blur-md"
            : "bg-transparent backdrop-blur-lg"
        }`}
        style={{
          animation: 'megaMenuFadeIn 0.2s ease-out',
        }}
      >
        <div className="grid grid-cols-12 gap-8">
          {/* Colunas de links */}
          {data.columns.map((column, colIndex) => (
            <div
              key={colIndex}
              className={`col-span-12 md:col-span-4 ${
                colIndex > 0 ? 'border-l border-white/10 pl-8' : ''
              }`}
            >
              <h3 className="text-xs font-bold uppercase tracking-wider text-white/60 mb-4">
                {column.title}
              </h3>
              <ul className="space-y-2">
                {column.items.map((item, itemIndex) => (
                  <li key={itemIndex}>
                    <Link
                      to={item.href}
                      className="block text-white/90 hover:text-white hover:translate-x-1 transition-all duration-150 text-sm py-1.5"
                      onClick={onClose}
                    >
                      {item.label}
                    </Link>
                  </li>
                ))}
              </ul>
            </div>
          ))}

          {/* Cards laterais */}
          <div className="col-span-12 md:col-span-4 border-l border-white/10 pl-8 space-y-4">
            {data.cards.map((card, cardIndex) => (
              <Link
                key={cardIndex}
                to={card.href}
                className="block bg-white/5 hover:bg-white/10 border border-white/10 rounded-xl p-6 transition-all duration-200 hover:border-white/20 group"
                onClick={onClose}
              >
                <h4 className="text-white font-semibold mb-2 group-hover:text-brand-red transition-colors">
                  {card.title}
                </h4>
                <p className="text-white/70 text-sm leading-relaxed">
                  {card.description}
                </p>
              </Link>
            ))}
          </div>
        </div>
      </div>
    </div>
  );
};

export default MegaMenu;
