import { useRef, useEffect } from 'react';
import { Link, useLocation } from 'react-router-dom';
import MegaMenu from './MegaMenu';
import type { NavItem } from './menuData';

interface MegaMenuNavItemProps {
  itemKey: string;
  item: NavItem;
  isOpen: boolean;
  onOpen: (key: string) => void;
  onClose: () => void;
  getNavItemClass: (path: string, hasMega: boolean) => string;
  location: ReturnType<typeof useLocation>;
  isHeaderSolid?: boolean;
}

const MegaMenuNavItem = ({
  itemKey,
  item,
  isOpen,
  onOpen,
  onClose,
  getNavItemClass,
  location,
  isHeaderSolid = false,
}: MegaMenuNavItemProps) => {
  const containerRef = useRef<HTMLDivElement>(null);
  const timeoutRef = useRef<NodeJS.Timeout | null>(null);
  const megaMenuContainerRef = useRef<HTMLDivElement>(null);

  const handleMouseEnter = () => {
    if (timeoutRef.current) {
      clearTimeout(timeoutRef.current);
      timeoutRef.current = null;
    }
    onOpen(itemKey);
  };

  const handleMouseLeave = () => {
    timeoutRef.current = setTimeout(() => {
      onClose();
    }, 150);
  };

  // Limpar timeout ao desmontar
  useEffect(() => {
    return () => {
      if (timeoutRef.current) {
        clearTimeout(timeoutRef.current);
      }
    };
  }, []);

  return (
    <div
      ref={containerRef}
      className="relative"
      onMouseEnter={handleMouseEnter}
      onMouseLeave={handleMouseLeave}
      onFocus={() => onOpen(itemKey)}
      onBlur={(e) => {
        // Fechar apenas se o foco não estiver no mega menu
        if (!e.currentTarget.contains(e.relatedTarget as Node)) {
          handleMouseLeave();
        }
      }}
    >
      <Link
        to={item.href}
        onClick={onClose}
        className={getNavItemClass(item.href, true)}
        aria-haspopup="menu"
        aria-expanded={isOpen}
      >
        {item.label}
      </Link>

      {item.mega && isOpen && (
        <div
          ref={megaMenuContainerRef}
          onMouseEnter={() => {
            if (timeoutRef.current) {
              clearTimeout(timeoutRef.current);
              timeoutRef.current = null;
            }
          }}
          onMouseLeave={handleMouseLeave}
        >
          <MegaMenu
            data={item.mega}
            isOpen={isOpen}
            onClose={onClose}
            triggerRef={containerRef}
            isHeaderSolid={isHeaderSolid}
          />
        </div>
      )}
    </div>
  );
};

export default MegaMenuNavItem;
