import { useState, useRef } from 'react';
import { Link, useLocation } from 'react-router-dom';
import { Menu, X, ChevronDown, ArrowLeft } from 'lucide-react';
import MegaMenuNavItem from './MegaMenuNavItem';
import LanguageSwitcher from '@/components/layout/LanguageSwitcher';
import { MENU_DATA, SIMPLE_NAV_ITEMS } from './menuData';

interface NavbarProps {
  isScrolled: boolean;
  isHomePage: boolean;
}

const Navbar = ({ isScrolled, isHomePage }: NavbarProps) => {
  const location = useLocation();
  const isHeaderSolid = isScrolled || !isHomePage;
  const [openMegaMenu, setOpenMegaMenu] = useState<string | null>(null);
  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);
  const [openMobileAccordion, setOpenMobileAccordion] = useState<string | null>(null);
  const timeoutRef = useRef<NodeJS.Timeout | null>(null);

  // Controlar abertura/fechamento do mega menu baseado no hover
  const megaMenuItems = ['servicos', 'eventos'];
  
  const handleMegaMenuOpen = (key: string) => {
    if (timeoutRef.current) {
      clearTimeout(timeoutRef.current);
      timeoutRef.current = null;
    }
    if (megaMenuItems.includes(key)) {
      setOpenMegaMenu(key);
    }
  };

  const handleMegaMenuClose = () => {
    if (timeoutRef.current) {
      clearTimeout(timeoutRef.current);
    }
    timeoutRef.current = setTimeout(() => {
      setOpenMegaMenu(null);
    }, 150);
  };

  const getNavItemClass = (path: string, hasMega: boolean = false) => {
    const isActive = location.pathname === path;
    const baseClass = 'px-4 py-2 rounded-full text-base font-medium transition-all duration-150';
    
    if (isActive) {
      return `${baseClass} bg-brand-red text-white shadow-md`;
    }
    
    if (isHomePage && !isScrolled) {
      return `${baseClass} text-white hover:bg-white/20 hover:text-white hover:shadow-md`;
    }
    
    return `${baseClass} text-white hover:bg-brand-red hover:text-white hover:shadow-md`;
  };

  return (
    <>
      {/* Desktop Navigation */}
      <div className="hidden lg:flex items-center space-x-1 relative">
        {!isHomePage && (
          <Link
            to="/"
            onClick={() => window.scrollTo({ top: 0, behavior: 'smooth' })}
            className={getNavItemClass('/', false)}
            aria-label="Voltar para o menu principal"
          >
            <ArrowLeft className="w-4 h-4" />
          </Link>
        )}
        {SIMPLE_NAV_ITEMS.map((item) => (
          <Link
            key={item.href}
            to={item.href}
            className={getNavItemClass(item.href)}
            style={{
              outline: location.pathname === item.href ? '2px solid hsl(var(--brand-red-focus))' : 'none',
              outlineOffset: '2px',
            }}
          >
            {item.label}
          </Link>
        ))}

        {/* Items com Mega Menu ou Links Simples */}
        {Object.entries(MENU_DATA).map(([key, item]) => {
          // Se não tem mega menu, renderiza como link simples
          if (!item.mega) {
            return (
              <Link
                key={key}
                to={item.href}
                className={getNavItemClass(item.href)}
                style={{
                  outline: location.pathname === item.href ? '2px solid hsl(var(--brand-red-focus))' : 'none',
                  outlineOffset: '2px',
                }}
              >
                {item.label}
              </Link>
            );
          }
          // Se tem mega menu, renderiza com MegaMenuNavItem
          return (
            <MegaMenuNavItem
              key={key}
              itemKey={key}
              item={item}
              isOpen={openMegaMenu === key}
              onOpen={handleMegaMenuOpen}
              onClose={handleMegaMenuClose}
              getNavItemClass={getNavItemClass}
              location={location}
              isHeaderSolid={isHeaderSolid}
            />
          );
        })}
      </div>

      {/* Mobile Menu Button */}
      <button
        className="lg:hidden p-2 transition-colors text-white"
        onClick={() => setIsMobileMenuOpen(!isMobileMenuOpen)}
        aria-label="Menu"
      >
        {isMobileMenuOpen ? <X className="w-6 h-6" /> : <Menu className="w-6 h-6" />}
      </button>

      {/* Mobile Menu */}
      {isMobileMenuOpen && (
        <div className="lg:hidden py-4 border-t border-white/20 animate-fade-in">
          {/* Language Switcher Mobile */}
          <div className="px-4 py-3 mb-2 border-b border-white/20">
            <LanguageSwitcher isWhite={true} />
          </div>

          {/* Items simples */}
          {SIMPLE_NAV_ITEMS.map((item) => (
            <Link
              key={item.href}
              to={item.href}
              onClick={() => setIsMobileMenuOpen(false)}
              className={`block mx-4 mb-2 px-4 py-3 rounded-full text-base font-medium transition-all duration-150 ${
                location.pathname === item.href
                  ? 'bg-brand-red text-white shadow-md'
                  : 'text-white hover:bg-white/20 hover:text-white hover:shadow-md'
              }`}
            >
              {item.label}
            </Link>
          ))}

          {/* Items com Accordion ou Links Simples */}
          {Object.entries(MENU_DATA).map(([key, item]) => {
            // Se não tem mega menu, renderiza como link simples
            if (!item.mega) {
              return (
                <Link
                  key={key}
                  to={item.href}
                  onClick={() => setIsMobileMenuOpen(false)}
                  className={`block mx-4 mb-2 px-4 py-3 rounded-full text-sm font-medium transition-all duration-150 ${
                    location.pathname === item.href
                      ? 'bg-brand-red text-white shadow-md'
                      : 'text-white hover:bg-white/20 hover:text-white hover:shadow-md'
                  }`}
                >
                  {item.label}
                </Link>
              );
            }
            // Se tem mega menu, renderiza com accordion
            return (
              <div key={key} className="mx-4 mb-2">
                <button
                  onClick={() =>
                    setOpenMobileAccordion(openMobileAccordion === key ? null : key)
                  }
                  className="flex items-center justify-between w-full px-4 py-3 rounded-full text-base font-medium text-white hover:bg-white/20 transition-all duration-150"
                >
                  <span>{item.label}</span>
                  <ChevronDown
                    className={`w-4 h-4 transition-transform ${
                      openMobileAccordion === key ? 'rotate-180' : ''
                    }`}
                  />
                </button>

                {openMobileAccordion === key && item.mega && (
                <div className="mt-2 ml-4 pl-4 border-l-2 border-white/20 space-y-1 animate-in slide-in-from-top-2">
                  {item.mega.columns.map((column, colIndex) => (
                    <div key={colIndex} className="mb-4">
                      <h4 className="text-xs font-bold uppercase tracking-wider text-white/60 mb-2">
                        {column.title}
                      </h4>
                      {column.items.map((subItem, itemIndex) => (
                        <Link
                          key={itemIndex}
                          to={subItem.href}
                          onClick={() => {
                            setIsMobileMenuOpen(false);
                            setOpenMobileAccordion(null);
                          }}
                          className="block text-white/80 hover:text-white py-1.5 text-base"
                        >
                          {subItem.label}
                        </Link>
                      ))}
                    </div>
                  ))}
                  {item.mega.cards.map((card, cardIndex) => (
                    <Link
                      key={cardIndex}
                      to={card.href}
                      onClick={() => {
                        setIsMobileMenuOpen(false);
                        setOpenMobileAccordion(null);
                      }}
                      className="block bg-white/5 border border-white/10 rounded-lg p-4 mt-4"
                    >
                      <h4 className="text-white font-semibold mb-1 text-sm">{card.title}</h4>
                      <p className="text-white/70 text-xs">{card.description}</p>
                    </Link>
                  ))}
                </div>
              )}
              </div>
            );
          })}
        </div>
      )}
    </>
  );
};

export default Navbar;
