import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { ExternalLink } from "lucide-react";

export interface CertificationInfo {
  title: string;
  what: string;
  when: string;
  who: string;
  validity?: string;
}

interface CertificationModalProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  certification: CertificationInfo | null;
}

export const certificationData: Record<string, CertificationInfo> = {
  "adagro": {
    title: "ADAGRO",
    what: "Agência de Defesa e Fiscalização Agropecuária de Pernambuco. Fiscaliza atividades relacionadas a produtos de origem animal e vegetal, incluindo serviços de controle de pragas.",
    when: "Exigida para empresas que atuam com produtos agropecuários, controle de pragas em estabelecimentos que lidam com alimentos e instalações rurais.",
    who: "Emitido pela ADAGRO - órgão estadual vinculado ao Governo de Pernambuco.",
    validity: "Renovação anual conforme regulamentação estadual."
  },
  "ibama": {
    title: "IBAMA",
    what: "Instituto Brasileiro do Meio Ambiente e dos Recursos Naturais Renováveis. Responsável por licenciamento ambiental federal e controle de produtos químicos.",
    when: "Necessário para empresas que utilizam produtos químicos controlados, atuam em áreas de preservação ou realizam atividades com impacto ambiental significativo.",
    who: "Emitido pelo IBAMA - autarquia federal do Ministério do Meio Ambiente.",
    validity: "Conforme tipo de licença e atividade desenvolvida."
  },
  "vigilancia": {
    title: "Vigilância Sanitária",
    what: "Órgão responsável por fiscalizar estabelecimentos que afetam a saúde pública, incluindo empresas de controle de pragas, limpeza de reservatórios e sanitização.",
    when: "Exigida para prestadores de serviços em ambientes de alimentação, saúde, educação e áreas com circulação de pessoas.",
    who: "Emitido pela Secretaria Municipal de Saúde por meio da Vigilância Sanitária local.",
    validity: "Renovação periódica mediante inspeção e adequação às normas sanitárias."
  },
  "dircon": {
    title: "DIRCON",
    what: "Licença municipal de funcionamento que autoriza o estabelecimento comercial ou industrial a operar no município. Abrange aspectos sanitários, de uso do solo e segurança.",
    when: "Obrigatória para todos os estabelecimentos comerciais e prestadores de serviço, incluindo empresas de controle de pragas.",
    who: "Emitido pela Prefeitura Municipal através da Diretoria de Controle Urbano (DIRCON) ou órgão equivalente.",
    validity: "Renovação anual, geralmente vinculada ao pagamento de taxas municipais."
  },
  "seman": {
    title: "SEMAN",
    what: "Secretaria de Meio Ambiente (SEMAN), responsável por autorizações e controle ambiental em âmbito municipal.",
    when: "Exigida para atividades que envolvam impacto ambiental local ou uso de produtos controlados.",
    who: "Emitido pela Secretaria Municipal de Meio Ambiente.",
    validity: "Conforme o tipo de licença ou autorização emitida."
  },
  "cprh": {
    title: "CPRH",
    what: "Agência Estadual de Meio Ambiente de Pernambuco. Responsável pelo licenciamento ambiental estadual e controle de atividades potencialmente poluidoras.",
    when: "Exigida para empreendimentos com impacto ambiental local, incluindo empresas de controle de pragas que manipulam produtos químicos.",
    who: "Emitido pela CPRH - órgão vinculado à Secretaria de Meio Ambiente de PE.",
    validity: "Renovação periódica conforme tipo de licença (prévia, instalação ou operação)."
  },
  "anvisa": {
    title: "ANVISA",
    what: "Agência Nacional de Vigilância Sanitária, responsável por regulamentar e fiscalizar produtos e serviços sujeitos ao controle sanitário em âmbito nacional.",
    when: "Necessária quando há uso de produtos sujeitos a controle sanitário nacional ou cumprimento de normas específicas da ANVISA.",
    who: "Emitido pela ANVISA ou por meio de autorizações, registros e normas por ela regulamentadas.",
    validity: "Conforme o tipo de autorização ou registro sanitário."
  },
  "crq": {
    title: "CRQ",
    what: "Conselho Regional de Química. Certifica a responsabilidade técnica de profissionais químicos em empresas que manipulam produtos químicos.",
    when: "Exigida quando há uso, manipulação ou comercialização de produtos químicos, incluindo inseticidas e desinfetantes.",
    who: "Emitido pelo CRQ - autarquia federal que fiscaliza o exercício da profissão de químico.",
    validity: "Anuidade profissional e registro da empresa devem estar em dia."
  },
  "nr-31": {
    title: "NR-31",
    what: "Norma Regulamentadora 31, que trata de segurança e saúde no trabalho na agricultura, pecuária, silvicultura, exploração florestal e aquicultura.",
    when: "Aplicável a serviços realizados em ambientes rurais ou agropecuários que envolvam risco ocupacional.",
    who: "Estabelecida pelo Ministério do Trabalho e Emprego.",
    validity: "Conformidade contínua com treinamentos e documentação atualizados."
  },
  "nr-33": {
    title: "NR-33",
    what: "Norma Regulamentadora 33, que estabelece requisitos para trabalhos em espaços confinados.",
    when: "Aplicável a serviços que envolvem entrada, permanência ou atividades em espaços confinados.",
    who: "Estabelecida pelo Ministério do Trabalho e Emprego.",
    validity: "Conformidade contínua com treinamentos e documentação atualizados."
  },
  "nr-35": {
    title: "NR-35",
    what: "Norma Regulamentadora 35, que trata de segurança para trabalhos em altura.",
    when: "Aplicável a serviços realizados acima de 2 metros de altura, com risco de queda.",
    who: "Estabelecida pelo Ministério do Trabalho e Emprego.",
    validity: "Conformidade contínua com treinamentos e documentação atualizados."
  }
};

const CertificationModal = ({ open, onOpenChange, certification }: CertificationModalProps) => {
  if (!certification) return null;

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-2xl max-h-[80vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="text-2xl font-bold text-primary">{certification.title}</DialogTitle>
        </DialogHeader>
        
        <div className="space-y-4 py-4">
          <div>
            <h3 className="font-semibold text-lg mb-2">O que é?</h3>
            <p className="text-muted-foreground">{certification.what}</p>
          </div>

          <div>
            <h3 className="font-semibold text-lg mb-2">Quando é exigido?</h3>
            <p className="text-muted-foreground">{certification.when}</p>
          </div>

          <div>
            <h3 className="font-semibold text-lg mb-2">Quem emite?</h3>
            <p className="text-muted-foreground">{certification.who}</p>
          </div>

          {certification.validity && (
            <div>
              <h3 className="font-semibold text-lg mb-2">Validade</h3>
              <p className="text-muted-foreground">{certification.validity}</p>
            </div>
          )}
        </div>

        <div className="pt-4 border-t">
          <Button 
            variant="default" 
            className="w-full" 
            asChild
          >
            <a
              href="https://wa.me/5581999999999?text=Olá!%20Gostaria%20de%20saber%20mais%20sobre%20certificações%20e%20conformidades."
              target="_blank"
              rel="noopener noreferrer"
            >
              <ExternalLink className="w-4 h-4 mr-2" />
              Fale com a Hecol
            </a>
          </Button>
        </div>
      </DialogContent>
    </Dialog>
  );
};

export default CertificationModal;