import { useState, useEffect } from "react";
import { Button } from "@/components/ui/button";
import { Settings, X } from "lucide-react";
import { useLanguage } from "@/lib/i18n/LanguageContext";
import CookiePreferencesModal from "./CookiePreferencesModal";
import { useCookieConsent } from "@/hooks/useCookieConsent";

const CookieBanner = () => {
  const [showPreferences, setShowPreferences] = useState(false);
  const { t } = useLanguage();
  const { showBanner, setShowBanner, acceptAll, rejectAll, savePreferences, openPreferences } = useCookieConsent();

  // Escuta evento global para abrir preferências
  useEffect(() => {
    const handleOpenPreferences = () => {
      openPreferences();
      setShowPreferences(true);
    };
    
    window.addEventListener('openCookiePreferences', handleOpenPreferences);
    return () => window.removeEventListener('openCookiePreferences', handleOpenPreferences);
  }, [openPreferences]);

  if (!showBanner) return null;

  return (
    <>
      {/* Top Banner - empurra conteúdo para baixo */}
      <div 
        className="fixed top-0 left-0 right-0 z-50 bg-background border-b border-border shadow-lg motion-safe:animate-in motion-safe:slide-in-from-top-2"
        role="dialog"
        aria-live="polite"
        aria-label={t.cookies.bannerTitle}
      >
        <div className="container mx-auto px-4 py-4">
          <div className="flex flex-col lg:flex-row items-start lg:items-center justify-between gap-4">
            <div className="flex-1 pr-4">
              <h2 className="text-lg font-semibold text-foreground mb-2">
                {t.cookies.bannerTitle}
              </h2>
              <p className="text-sm text-muted-foreground leading-relaxed">
                {t.cookies.bannerDescription}
              </p>
            </div>
            
            <div className="flex flex-wrap gap-2 items-center w-full lg:w-auto justify-end">
              <Button
                variant="outline"
                size="sm"
                onClick={() => setShowPreferences(true)}
                className="text-sm"
                aria-label={t.cookies.customizeButton}
              >
                <Settings className="w-4 h-4 mr-2" />
                {t.cookies.customizeButton}
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => {
                  rejectAll();
                  setShowBanner(false);
                }}
                className="text-sm"
                aria-label={t.cookies.rejectAll}
              >
                {t.cookies.rejectAll}
              </Button>
              <Button
                size="sm"
                onClick={() => {
                  acceptAll();
                  setShowBanner(false);
                }}
                className="text-sm bg-brand-red hover:bg-brand-red/90 text-white"
                aria-label={t.cookies.acceptAll}
              >
                {t.cookies.acceptAll}
              </Button>
            </div>
          </div>
        </div>
      </div>

      {/* Spacer para empurrar o conteúdo */}
      <div className="h-[120px] lg:h-[88px]" aria-hidden="true" />

      <CookiePreferencesModal
        open={showPreferences}
        onOpenChange={setShowPreferences}
        onSave={(prefs) => {
          savePreferences(prefs);
          setShowPreferences(false);
        }}
      />
    </>
  );
};

export default CookieBanner;
