import { useState, useEffect } from "react";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";
import { useLanguage } from "@/lib/i18n/LanguageContext";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Separator } from "@/components/ui/separator";

interface CookiePreferencesModalProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  onSave: (preferences: { 
    preferences: boolean; 
    analytics: boolean; 
    marketing: boolean;
  }) => void;
}

const CookiePreferencesModal = ({
  open,
  onOpenChange,
  onSave,
}: CookiePreferencesModalProps) => {
  const { t } = useLanguage();
  const [preferences, setPreferences] = useState(false);
  const [analytics, setAnalytics] = useState(false);
  const [marketing, setMarketing] = useState(false);

  // Reset ao abrir
  useEffect(() => {
    if (open) {
      setPreferences(false);
      setAnalytics(false);
      setMarketing(false);
    }
  }, [open]);

  const handleSave = () => {
    onSave({ preferences, analytics, marketing });
  };

  const handleAcceptAll = () => {
    setPreferences(true);
    setAnalytics(true);
    setMarketing(true);
    onSave({ preferences: true, analytics: true, marketing: true });
  };

  const handleRejectAll = () => {
    setPreferences(false);
    setAnalytics(false);
    setMarketing(false);
    onSave({ preferences: false, analytics: false, marketing: false });
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-[600px] max-h-[90vh]">
        <DialogHeader>
          <DialogTitle className="text-xl">{t.cookies.modalTitle}</DialogTitle>
          <DialogDescription className="text-sm">
            {t.cookies.modalDescription}
          </DialogDescription>
        </DialogHeader>

        <ScrollArea className="max-h-[60vh] pr-4">
          <div className="space-y-6 py-4">
            {/* Necessários */}
            <div className="space-y-3">
              <div className="flex items-start justify-between space-x-4">
                <div className="flex-1">
                  <Label className="text-base font-semibold flex items-center gap-2">
                    {t.cookies.necessary}
                    <span className="text-xs font-normal bg-muted px-2 py-0.5 rounded">
                      {t.cookies.alwaysActive}
                    </span>
                  </Label>
                  <p className="text-sm text-muted-foreground mt-2 leading-relaxed">
                    {t.cookies.necessaryDesc}
                  </p>
                  <p className="text-xs text-muted-foreground mt-2 italic">
                    {t.cookies.necessaryExamples}
                  </p>
                </div>
                <Switch checked disabled className="mt-1 data-[state=checked]:bg-brand-red" />
              </div>
              <Separator />
            </div>

            {/* Preferências */}
            <div className="space-y-3">
              <div className="flex items-start justify-between space-x-4">
                <div className="flex-1">
                  <Label htmlFor="preferences" className="text-base font-semibold cursor-pointer">
                    {t.cookies.preferences}
                  </Label>
                  <p className="text-sm text-muted-foreground mt-2 leading-relaxed">
                    {t.cookies.preferencesDesc}
                  </p>
                  <p className="text-xs text-muted-foreground mt-2 italic">
                    {t.cookies.preferencesExamples}
                  </p>
                </div>
                <Switch
                  id="preferences"
                  checked={preferences}
                  onCheckedChange={setPreferences}
                  className="mt-1 data-[state=checked]:bg-brand-red"
                />
              </div>
              <Separator />
            </div>

            {/* Analytics */}
            <div className="space-y-3">
              <div className="flex items-start justify-between space-x-4">
                <div className="flex-1">
                  <Label htmlFor="analytics" className="text-base font-semibold cursor-pointer">
                    {t.cookies.analytics}
                  </Label>
                  <p className="text-sm text-muted-foreground mt-2 leading-relaxed">
                    {t.cookies.analyticsDesc}
                  </p>
                  <p className="text-xs text-muted-foreground mt-2 italic">
                    {t.cookies.analyticsExamples}
                  </p>
                </div>
                <Switch
                  id="analytics"
                  checked={analytics}
                  onCheckedChange={setAnalytics}
                  className="mt-1 data-[state=checked]:bg-brand-red"
                />
              </div>
              <Separator />
            </div>

            {/* Marketing */}
            <div className="space-y-3">
              <div className="flex items-start justify-between space-x-4">
                <div className="flex-1">
                  <Label htmlFor="marketing" className="text-base font-semibold cursor-pointer">
                    {t.cookies.marketing}
                  </Label>
                  <p className="text-sm text-muted-foreground mt-2 leading-relaxed">
                    {t.cookies.marketingDesc}
                  </p>
                  <p className="text-xs text-muted-foreground mt-2 italic">
                    {t.cookies.marketingExamples}
                  </p>
                </div>
                <Switch
                  id="marketing"
                  checked={marketing}
                  onCheckedChange={setMarketing}
                  className="mt-1 data-[state=checked]:bg-brand-red"
                />
              </div>
            </div>
          </div>
        </ScrollArea>

        <Separator />

        <div className="flex flex-col sm:flex-row justify-between gap-2 pt-2">
          <div className="flex gap-2">
            <Button 
              variant="outline" 
              onClick={handleRejectAll}
              className="text-sm"
            >
              {t.cookies.rejectAll}
            </Button>
            <Button 
              variant="outline" 
              onClick={handleAcceptAll}
              className="text-sm"
            >
              {t.cookies.acceptAll}
            </Button>
          </div>
          <Button 
            onClick={handleSave}
            className="bg-brand-red hover:bg-brand-red/90 text-white"
          >
            {t.cookies.savePreferences}
          </Button>
        </div>
      </DialogContent>
    </Dialog>
  );
};

export default CookiePreferencesModal;
