import { Button } from "@/components/ui/button";
import { Shield, Clock, Award, Briefcase, ChevronLeft, ChevronRight } from "lucide-react";
import { useLanguage } from "@/lib/i18n/LanguageContext";
import { useEffect, useState } from "react";
import { Link } from "react-router-dom";

const Hero = () => {
  const { t } = useLanguage();
  // Carregar as duas imagens diretamente da pasta hero
  const heroImages = [
    '/hero/hecol-equipe-fullbody.jpg',
    '/hero/antes-depois-03.jpg',
  ];
  const [currentIndex, setCurrentIndex] = useState(0);

  const goToSlide = (index: number) => {
    setCurrentIndex(index);
  };

  const goToPrevious = () => {
    setCurrentIndex((prev) => (prev - 1 + heroImages.length) % heroImages.length);
  };

  const goToNext = () => {
    setCurrentIndex((prev) => (prev + 1) % heroImages.length);
  };

  return (
    <section 
      className="relative overflow-hidden min-h-[90vh] lg:min-h-[100vh] flex items-center pt-0" 
      role="banner"
    >
      {/* Background Images Carousel */}
      <div className="absolute inset-0 z-0">
        {heroImages.map((image, index) => (
          <div
            key={index}
            className={`absolute inset-0 bg-cover bg-center bg-no-repeat transition-opacity duration-1000 ${
              index === currentIndex ? 'opacity-100' : 'opacity-0'
            }`}
            style={{
              backgroundImage: `url(${image})`,
              backgroundSize: 'cover',
              backgroundPosition: 'center',
            }}
            aria-hidden={index !== currentIndex}
          />
        ))}
      </div>

      {/* Dark Overlay Gradient */}
      <div
        className="absolute inset-0 z-10"
        style={{
          background: 'linear-gradient(180deg, rgba(0,0,0,0.55) 0%, rgba(0,0,0,0.45) 40%, rgba(0,0,0,0.7) 100%)',
        }}
        aria-hidden="true"
      />

      {/* Navigation Arrows - Sempre visíveis quando há mais de 1 imagem */}
      {heroImages.length > 1 && (
        <>
          <button
            onClick={goToPrevious}
            className="absolute left-4 top-1/2 -translate-y-1/2 z-50 p-4 rounded-full bg-white/10 hover:bg-white/20 backdrop-blur-md border border-white/30 transition-all duration-300 group shadow-lg hover:shadow-xl hover:scale-110"
            aria-label="Imagem anterior"
          >
            <ChevronLeft className="w-7 h-7 text-white group-hover:scale-110 transition-transform" />
          </button>
          <button
            onClick={goToNext}
            className="absolute right-4 top-1/2 -translate-y-1/2 z-50 p-4 rounded-full bg-white/10 hover:bg-white/20 backdrop-blur-md border border-white/30 transition-all duration-300 group shadow-lg hover:shadow-xl hover:scale-110"
            aria-label="Próxima imagem"
          >
            <ChevronRight className="w-7 h-7 text-white group-hover:scale-110 transition-transform" />
          </button>
        </>
      )}

      {/* Content */}
      <div className="container mx-auto px-4 relative z-20 pt-24 md:pt-28 lg:pt-32">
        <div className="grid lg:grid-cols-1 gap-12 items-center max-w-4xl">
          {/* Content Column */}
          <div className="max-w-2xl">
            <h1 className="text-4xl md:text-5xl lg:text-6xl font-bold text-white mb-6 animate-fade-in leading-tight">
              {t.hero.title.split(' ').slice(0, -2).join(' ')}{' '}
              <span className="text-primary-foreground">{t.hero.title.split(' ').slice(-2).join(' ')}</span>
            </h1>

            <p className="text-lg md:text-xl text-white/90 mb-8 leading-relaxed animate-fade-in-up">
              {t.hero.subtitle}
            </p>

            <div className="animate-fade-in-up mb-12 flex flex-wrap gap-4">
              <Button 
                size="lg" 
                variant="cta"
                asChild
              >
                <a href="/contato">
                  {t.hero.cta}
                </a>
              </Button>
              <Button 
                size="lg" 
                variant="cta"
                asChild
              >
                <Link to="/trabalhe-conosco">
                  <Briefcase className="w-5 h-5 mr-2" />
                  {t.nav.trabalheConosco}
                </Link>
              </Button>
            </div>

            {/* Trust Badges */}
            <div className="grid grid-cols-1 sm:grid-cols-3 gap-6 animate-scale-in">
              <div className="flex items-center gap-3 text-white/90">
                <div className="bg-white/10 backdrop-blur-sm p-3 rounded-lg border border-white/20">
                  <Shield className="w-6 h-6 text-white" />
                </div>
                <div>
                  <div className="font-semibold text-white">Certificações</div>
                  <div className="text-sm text-white/70">Anvisa e ADAGRO</div>
                </div>
              </div>
              <div className="flex items-center gap-3 text-white/90">
                <div className="bg-white/10 backdrop-blur-sm p-3 rounded-lg border border-white/20">
                  <Clock className="w-6 h-6 text-white" />
                </div>
                <div>
                  <div className="font-semibold text-white">Atendimento Rápido</div>
                  <div className="text-sm text-white/70">24/7 Disponível</div>
                </div>
              </div>
              <div className="flex items-center gap-3 text-white/90">
                <div className="bg-white/10 backdrop-blur-sm p-3 rounded-lg border border-white/20">
                  <Award className="w-6 h-6 text-white" />
                </div>
                <div>
                  <div className="font-semibold text-white">Experiência</div>
                  <div className="text-sm text-white/70">+15 Anos</div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Dots Indicator */}
      {heroImages.length > 1 && (
        <div className="absolute bottom-6 left-1/2 -translate-x-1/2 z-50 flex gap-2">
          {heroImages.map((_, index) => (
            <button
              key={index}
              onClick={() => goToSlide(index)}
              className={`h-2 rounded-full transition-all duration-300 ${
                index === currentIndex
                  ? 'w-8 bg-white'
                  : 'w-2 bg-white/50 hover:bg-white/70'
              }`}
              aria-label={`Ir para imagem ${index + 1}`}
            />
          ))}
        </div>
      )}
    </section>
  );
};

export default Hero;
