import { MapPin, Navigation, Phone } from "lucide-react";
import { Button } from "@/components/ui/button";
import { useLanguage } from "@/lib/i18n/LanguageContext";
import { useEffect, useState } from "react";
import { loadCompanyData, loadManifest, getAssetsBySection, getAssetPath, type CompanyData } from "@/lib/contentLoader";
import { getGoogleMapsDirectionsUrl } from "@/config/company";

const LocationSection = () => {
  const { t } = useLanguage();
  const [companyData, setCompanyData] = useState<CompanyData | null>(null);
  const [backgroundImage, setBackgroundImage] = useState("");

  useEffect(() => {
    loadCompanyData().then(data => setCompanyData(data));
    loadManifest().then(manifest => {
      const backgrounds = getAssetsBySection(manifest, 'location');
      if (backgrounds.length > 0) {
        setBackgroundImage(getAssetPath('location', backgrounds[0].filename));
      }
    });
  }, []);

  if (!companyData) return null;

  const { address } = companyData.company;

  const fullAddress = `${address.line1}, ${address.neighborhood}, ${address.city} - ${address.state}, ${address.postal_code}`;
  const mapUrl = `https://www.google.com/maps/embed/v1/place?key=AIzaSyBFw0Qbyq9zTFTd-tUY6dZWTgaQzuU17R8&q=${address.location.lat},${address.location.lng}&zoom=15`;

  return (
    <section 
      className="relative overflow-hidden bg-primary-dark border-t-2 border-brand-red-divider" 
      id="localizacao"
      aria-label={t.location.title}
    >
      {/* Background Image with Overlay */}
      {backgroundImage && (
        <div 
          className="absolute inset-0 bg-cover bg-center"
          style={{
            backgroundImage: `url(${backgroundImage})`,
            filter: 'brightness(0.3)',
          }}
          aria-hidden="true"
        />
      )}
      
      <div className="container mx-auto px-4 py-20 relative z-10">
        <div className="grid lg:grid-cols-2 gap-8 items-center">
          {/* Left side - Info */}
          <div className="text-white">
            <h2 className="text-3xl md:text-4xl font-bold mb-6 flex items-center gap-3">
              <MapPin className="w-8 h-8 text-primary-foreground" aria-hidden="true" />
              {t.location.title}
            </h2>
            
            <div className="bg-white/10 backdrop-blur-sm p-6 rounded-lg border border-white/20 mb-6">
              <p className="text-lg leading-relaxed text-white/90">
                {fullAddress}
              </p>
            </div>

            <div className="flex flex-col sm:flex-row gap-4">
              <Button 
                size="lg"
                variant="default"
                className="gap-2"
                asChild
              >
                <a 
                  href={getGoogleMapsDirectionsUrl()}
                  target="_blank"
                  rel="noopener noreferrer"
                  aria-label={`${t.location.route} - ${fullAddress}`}
                >
                  <Navigation className="w-5 h-5" aria-hidden="true" />
                  {t.location.route}
                </a>
              </Button>
              
              <Button 
                size="lg"
                variant="outline"
                className="gap-2 bg-white/10 border-white/30 text-white hover:text-brand-red-contrast"
                asChild
              >
                <a 
                  href={`tel:${companyData.company.phones[0]}`}
                  aria-label={`${t.location.contact} - ${companyData.company.phones[0]}`}
                >
                  <Phone className="w-5 h-5" aria-hidden="true" />
                  {t.location.contact}
                </a>
              </Button>
            </div>
          </div>

          {/* Right side - Map */}
          <div className="relative h-[400px] lg:h-[500px] rounded-lg overflow-hidden shadow-2xl border-2 border-white/20">
            <iframe
              src={mapUrl}
              width="100%"
              height="100%"
              style={{ border: 0 }}
              allowFullScreen
              loading="lazy"
              referrerPolicy="no-referrer-when-downgrade"
              title={`Mapa de localização - ${fullAddress}`}
              className="grayscale-[30%]"
            />
          </div>
        </div>
      </div>
    </section>
  );
};

export default LocationSection;
