import { useState, useEffect, useRef, RefObject } from 'react';

interface UseHoverIntentOptions {
  delay?: number; // Delay em ms antes de fechar
  sensitivity?: number; // Sensibilidade do movimento do mouse
}

/**
 * Hook para detectar hover com delay (evita flicker ao mover o mouse)
 */
export const useHoverIntent = (
  options: UseHoverIntentOptions = {}
): [boolean, RefObject<HTMLDivElement>, RefObject<HTMLDivElement>] => {
  const { delay = 150, sensitivity = 7 } = options;
  const [isHovered, setIsHovered] = useState(false);
  const timeoutRef = useRef<NodeJS.Timeout | null>(null);
  const elementRef = useRef<HTMLDivElement>(null);
  const triggerRef = useRef<HTMLDivElement>(null);
  const mousePositionRef = useRef({ x: 0, y: 0 });
  const isMouseMovingRef = useRef(false);

  useEffect(() => {
    const element = elementRef.current;
    const trigger = triggerRef.current;
    if (!element || !trigger) return;

    const handleMouseEnter = () => {
      if (timeoutRef.current) {
        clearTimeout(timeoutRef.current);
        timeoutRef.current = null;
      }
      setIsHovered(true);
    };

    const handleMouseLeave = () => {
      timeoutRef.current = setTimeout(() => {
        setIsHovered(false);
      }, delay);
    };

    const handleMouseMove = (e: MouseEvent) => {
      const { x, y } = mousePositionRef.current;
      const distance = Math.sqrt(
        Math.pow(e.clientX - x, 2) + Math.pow(e.clientY - y, 2)
      );

      if (distance > sensitivity) {
        isMouseMovingRef.current = true;
      }

      mousePositionRef.current = { x: e.clientX, y: e.clientY };
    };

    // Eventos no elemento do mega menu
    element.addEventListener('mouseenter', handleMouseEnter);
    element.addEventListener('mouseleave', handleMouseLeave);

    // Eventos no trigger (botão do menu)
    trigger.addEventListener('mouseenter', handleMouseEnter);
    trigger.addEventListener('mouseleave', handleMouseLeave);

    // Detectar movimento do mouse globalmente
    document.addEventListener('mousemove', handleMouseMove);

    return () => {
      element.removeEventListener('mouseenter', handleMouseEnter);
      element.removeEventListener('mouseleave', handleMouseLeave);
      trigger.removeEventListener('mouseenter', handleMouseEnter);
      trigger.removeEventListener('mouseleave', handleMouseLeave);
      document.removeEventListener('mousemove', handleMouseMove);
      if (timeoutRef.current) {
        clearTimeout(timeoutRef.current);
      }
    };
  }, [delay, sensitivity]);

  return [isHovered, triggerRef, elementRef];
};
