import Papa from 'papaparse';
import yaml from 'js-yaml';

export interface ManifestRow {
  section: string;
  filename: string;
  title_pt: string;
  alt_pt: string;
  title_en: string;
  alt_en: string;
  title_es: string;
  alt_es: string;
  order: string;
  link?: string;
  tags?: string;
}

export interface CompanyData {
  company: {
    name: string;
    cnpj: string;
    phones: string[];
    email: string;
    address: {
      line1: string;
      neighborhood: string;
      city: string;
      state: string;
      postal_code: string;
      country: string;
      location: {
        lat: number;
        lng: number;
      };
    };
    social: {
      instagram: string;
      linkedin: string;
      facebook: string;
    };
    website: {
      url: string;
      whatsapp_contact_url: string;
    };
  };
  numbers: {
    atendimentos_realizados: number;
    tempo_medio_resposta_min: number;
    anos_experiencia: number;
  };
  compliance: string[];
}

let manifestCache: ManifestRow[] | null = null;
let companyCache: CompanyData | null = null;

export async function loadManifest(): Promise<ManifestRow[]> {
  // Em desenvolvimento, sempre recarrega para pegar mudanças
  if (import.meta.env.PROD && manifestCache) return manifestCache;
  
  try {
    const response = await fetch('/content/manifest.csv?t=' + Date.now());
    if (!response.ok) return [];
    
    const csvText = await response.text();
    const result = Papa.parse<ManifestRow>(csvText, {
      header: true,
      skipEmptyLines: true,
    });
    
    manifestCache = result.data;
    return manifestCache;
  } catch (error) {
    console.warn('Failed to load manifest.csv:', error);
    return [];
  }
}

export async function loadCompanyData(): Promise<CompanyData | null> {
  if (companyCache) return companyCache;
  
  try {
    const response = await fetch('/content/company.yaml');
    if (!response.ok) return null;
    
    const yamlText = await response.text();
    companyCache = yaml.load(yamlText) as CompanyData;
    return companyCache;
  } catch (error) {
    console.warn('Failed to load company.yaml:', error);
    return null;
  }
}

export function getAssetsBySection(manifest: ManifestRow[], section: string): ManifestRow[] {
  return manifest
    .filter(row => row.section === section)
    .sort((a, b) => parseInt(a.order) - parseInt(b.order));
}

export function getAssetPath(section: string, filename: string): string {
  return `/${section}/${filename}`;
}
