import { useState } from "react";
import Header from "@/components/layout/Header";
import Footer from "@/components/layout/Footer";
import { Card } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Calendar, MapPin, Search } from "lucide-react";

interface Event {
  id: string;
  title: string;
  summary: string;
  start_date: string;
  location: string;
  cover_url: string | null;
}

// Dados estáticos de eventos
const staticEvents: Event[] = [
  {
    id: "1",
    title: "Treinamento NR-33 e NR-35",
    summary: "Capacitação em segurança para trabalhos em espaços confinados e em altura. Certificação reconhecida pelo MTE.",
    start_date: "2024-03-15",
    location: "Sede Hecol - Recife/PE",
    cover_url: "/gallery/curso1.jpg",
  },
  {
    id: "2",
    title: "Workshop de Boas Práticas em Controle de Pragas",
    summary: "Evento para síndicos e administradores de condomínios sobre prevenção e controle integrado de pragas urbanas.",
    start_date: "2024-04-20",
    location: "Hotel Grand Mercure - Recife/PE",
    cover_url: "/gallery/curso2.jpg",
  },
  {
    id: "3",
    title: "Feira de Saúde Ambiental",
    summary: "Participação na feira anual de saúde ambiental com exposição de técnicas e equipamentos modernos.",
    start_date: "2024-05-10",
    location: "Centro de Convenções - Olinda/PE",
    cover_url: null,
  },
];

const Eventos = () => {
  const [searchTerm, setSearchTerm] = useState("");

  const filteredEvents = staticEvents.filter((event) =>
    event.title.toLowerCase().includes(searchTerm.toLowerCase())
  );

  return (
    <div className="min-h-screen">
      <Header />
      <main>
        {/* Hero */}
        <section className="py-20 bg-gradient-to-br from-secondary to-secondary/80 text-white border-t-2 border-brand-red-divider">
          <div className="container mx-auto px-4 text-center">
            <h1 className="mb-6">Eventos e Treinamentos</h1>
            <p className="text-xl text-gray-200 max-w-3xl mx-auto">
              Fique por dentro das nossas ações, feiras, treinamentos e eventos de capacitação
              profissional.
            </p>
          </div>
        </section>

        {/* Busca */}
        <section className="py-8 bg-background border-t-2 border-brand-red-divider">
          <div className="container mx-auto px-4">
            <div className="max-w-md mx-auto relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
              <Input
                type="text"
                placeholder="Buscar eventos..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10"
              />
            </div>
          </div>
        </section>

        {/* Grid de Eventos */}
        <section className="py-16 bg-background">
          <div className="container mx-auto px-4">
            {filteredEvents.length > 0 ? (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {filteredEvents.map((event) => (
                  <Card
                    key={event.id}
                    className="overflow-hidden group hover:shadow-xl transition-all duration-300 border-brand-red-muted/30"
                  >
                    {event.cover_url && (
                      <div className="relative h-48 overflow-hidden">
                        <img
                          src={event.cover_url}
                          alt={event.title}
                          className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
                          loading="lazy"
                        />
                      </div>
                    )}
                    <div className="p-6">
                      <h3 className="font-bold text-xl mb-2">{event.title}</h3>
                      <div className="space-y-2 text-sm text-muted-foreground mb-4">
                        <div className="flex items-center">
                          <Calendar className="w-4 h-4 mr-2" />
                          {new Date(event.start_date).toLocaleDateString("pt-BR", {
                            day: "2-digit",
                            month: "long",
                            year: "numeric",
                          })}
                        </div>
                        <div className="flex items-center">
                          <MapPin className="w-4 h-4 mr-2" />
                          {event.location}
                        </div>
                      </div>
                      <p className="text-muted-foreground mb-4 line-clamp-3">{event.summary}</p>
                    </div>
                  </Card>
                ))}
              </div>
            ) : (
              <div className="text-center py-12">
                <p className="text-muted-foreground">
                  {staticEvents.length === 0
                    ? "Nenhum evento disponível no momento. Em breve disponibilizaremos novidades."
                    : "Nenhum evento encontrado com esses critérios."}
                </p>
              </div>
            )}
          </div>
        </section>

        {/* CTA */}
        <section className="py-16 bg-muted border-t-2 border-brand-red-divider">
          <div className="container mx-auto px-4 text-center">
            <h2 className="mb-6">Não Perca Nossos Próximos Eventos</h2>
            <p className="text-lg text-muted-foreground mb-8 max-w-2xl mx-auto">
              Entre em contato e receba informações sobre futuros eventos, treinamentos e ações
              sociais.
            </p>
            <Button variant="hero" size="lg" asChild>
              <a
                href="https://wa.me/5581999999999?text=Olá!%20Gostaria%20de%20receber%20informações%20sobre%20eventos."
                target="_blank"
                rel="noopener noreferrer"
              >
                Receber Novidades
              </a>
            </Button>
          </div>
        </section>
      </main>
      <Footer />
    </div>
  );
};

export default Eventos;
