import { useState } from "react";
import Header from "@/components/layout/Header";
import Footer from "@/components/layout/Footer";
import { Card } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Search } from "lucide-react";

interface GalleryImage {
  id: string;
  title: string;
  alt: string;
  category: string;
  image_url: string;
}

// Dados estáticos da galeria
const staticImages: GalleryImage[] = [
  {
    id: "1",
    title: "Equipe em Ação",
    alt: "Equipe Hecol em serviço",
    category: "equipe",
    image_url: "/gallery/andando.jpg",
  },
  {
    id: "2",
    title: "Antes e Depois - Trabalho 1",
    alt: "Antes e depois do serviço de desinsetização",
    category: "antes-depois",
    image_url: "/gallery/antes-depois-01.jpg",
  },
  {
    id: "3",
    title: "Antes e Depois - Trabalho 2",
    alt: "Antes e depois do serviço de limpeza",
    category: "antes-depois",
    image_url: "/gallery/antes-depois-02.jpg",
  },
  {
    id: "4",
    title: "Antes e Depois - Trabalho 3",
    alt: "Antes e depois do serviço de sanitização",
    category: "antes-depois",
    image_url: "/gallery/antes-depois-03.jpg",
  },
  {
    id: "5",
    title: "Treinamento Profissional",
    alt: "Curso de capacitação da equipe",
    category: "treinamento",
    image_url: "/gallery/curso1.jpg",
  },
  {
    id: "6",
    title: "Certificação NR",
    alt: "Treinamento de normas regulamentadoras",
    category: "treinamento",
    image_url: "/gallery/curso2.jpg",
  },
  {
    id: "7",
    title: "Instalação Hospitalar",
    alt: "Serviço em ambiente hospitalar",
    category: "servicos",
    image_url: "/gallery/instalando-hospital.jpg",
  },
  {
    id: "8",
    title: "Equipe Hecol",
    alt: "Profissionais da Hecol",
    category: "equipe",
    image_url: "/gallery/pose-de-frente.jpg",
  },
];

const Galeria = () => {
  const [selectedCategory, setSelectedCategory] = useState("todas");
  const [searchTerm, setSearchTerm] = useState("");

  const categories = ["todas", ...Array.from(new Set(staticImages.map((img) => img.category)))];

  const filteredImages = staticImages.filter((img) => {
    const matchesCategory = selectedCategory === "todas" || img.category === selectedCategory;
    const matchesSearch = img.title.toLowerCase().includes(searchTerm.toLowerCase());
    return matchesCategory && matchesSearch;
  });

  const getCategoryLabel = (category: string) => {
    const labels: Record<string, string> = {
      todas: "Todas",
      "antes-depois": "Antes e Depois",
      equipe: "Equipe",
      treinamento: "Treinamentos",
      servicos: "Serviços",
    };
    return labels[category] || category;
  };

  return (
    <div className="min-h-screen">
      <Header />
      <main>
        {/* Hero */}
        <section className="pt-24 pb-20 bg-gradient-to-br from-secondary to-secondary/80 text-white">
          <div className="container mx-auto px-4 text-center">
            <h1 className="mb-6">Galeria de Trabalhos</h1>
            <p className="text-xl text-gray-200 max-w-3xl mx-auto">
              Veja nossos serviços realizados e a qualidade do nosso trabalho.
            </p>
          </div>
        </section>

        {/* Filtros e Busca */}
        <section className="py-8 bg-background border-t-2 border-brand-red-divider">
          <div className="container mx-auto px-4">
            <div className="flex flex-col md:flex-row gap-4 items-center justify-between">
              {/* Categorias */}
              <div className="flex gap-2 flex-wrap">
                {categories.map((category) => (
                  <Button
                    key={category}
                    variant={selectedCategory === category ? "default" : "outline"}
                    onClick={() => setSelectedCategory(category)}
                  >
                    {getCategoryLabel(category)}
                  </Button>
                ))}
              </div>

              {/* Busca */}
              <div className="relative w-full md:w-auto md:min-w-[300px]">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
                <Input
                  type="text"
                  placeholder="Buscar imagens..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="pl-10"
                />
              </div>
            </div>
          </div>
        </section>

        {/* Grid de Imagens */}
        <section className="py-16 bg-background">
          <div className="container mx-auto px-4">
            {filteredImages.length > 0 ? (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {filteredImages.map((image) => (
                  <Card
                    key={image.id}
                    className="overflow-hidden group cursor-pointer hover:shadow-xl transition-all duration-300 border-brand-red-muted/30"
                  >
                    <div className="relative h-64 overflow-hidden">
                      <img
                        src={image.image_url}
                        alt={image.alt}
                        className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
                        loading="lazy"
                      />
                    </div>
                    <div className="p-4">
                      <h3 className="font-bold text-lg mb-2">{image.title}</h3>
                      <span className="text-sm text-muted-foreground capitalize">
                        {image.category}
                      </span>
                    </div>
                  </Card>
                ))}
              </div>
            ) : (
              <div className="text-center py-12">
                <p className="text-muted-foreground">
                  Nenhuma imagem encontrada com esses critérios.
                </p>
              </div>
            )}
          </div>
        </section>

        {/* CTA */}
        <section className="py-16 bg-muted border-t-2 border-brand-red-divider">
          <div className="container mx-auto px-4 text-center">
            <h2 className="mb-6">Quer Resultados Como Estes?</h2>
            <p className="text-lg text-muted-foreground mb-8 max-w-2xl mx-auto">
              Entre em contato e veja como podemos transformar seu ambiente.
            </p>
            <Button variant="hero" size="lg" asChild>
              <a
                href="https://wa.me/5581999999999?text=Olá!%20Gostaria%20de%20solicitar%20um%20orçamento."
                target="_blank"
                rel="noopener noreferrer"
              >
                Solicitar Orçamento
              </a>
            </Button>
          </div>
        </section>
      </main>
      <Footer />
    </div>
  );
};

export default Galeria;
